package serveur;

import java.io.IOException;
import java.net.MalformedURLException;
import java.rmi.NotBoundException;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;
import java.rmi.server.UnicastRemoteObject;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.Vector;

import javax.naming.NamingException;

import serveur.exceptions.ServeurCommunicationException;
import serveur.exceptions.ServeurException;
import serveur.exceptions.ServeurRefuseTraitementException;
import tools.Loggeur;

import client.ClientInfos;
import client.ClientRemote;
import decoupage.Decoupage;
import decoupage.ImageSerializable;
import decoupage.Morceau;
import filtre.EnumTypeFiltre;

import admin.AdminInfos;
import admin.AdminRemote;

public class Serveur extends UnicastRemoteObject implements ServeurRemote, Runnable {

	public static final int MIN_THREAD = 2;
	public static final int MAX_THREAD = 4;

	private ServeurInfos serveurInfos;
	private ServeurRemote suivant;

	private SafraTerminaison terminaison;

	private ClientInfos clientEnCours;
	private boolean enCoursTraitement;

	private AdminInfos adminInfos;
	
	private Thread serveurJeton;
	private LinkedList<Jeton> fileJeton;

	private Vector<ThreadTraitement> tabThreads;

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * Constructeur qui prend les infos du serveur
	 * @param serveur Les infos du serveur
	 * @throws RemoteException
	 */
	public Serveur(ServeurInfos infos) throws RemoteException {
		super();
		this.serveurInfos = infos;
		suivant = null;
		clientEnCours = null;
		adminInfos = null;
		enCoursTraitement = false;
		serveurJeton = new Thread(this);
		fileJeton = new LinkedList<Jeton>();
		terminaison = new SafraTerminaison(this);
	}

	/**
	 * Effectue l'initialisation du serveur (RMI, création des threads de traitements...)
	 * @throws RemoteException
	 */
	public void init() throws RemoteException {
		Loggeur.writeLog("Serveur", "Initialisation");
		Loggeur.writeLog("Serveur", "Bind du serveur " + serveurInfos);
		Registry rServeur = LocateRegistry.getRegistry("localhost", serveurInfos.port);
		rServeur.rebind(serveurInfos.nom, this);

		int nbThreads = new Random().nextInt(MAX_THREAD - MIN_THREAD + 1) + MIN_THREAD;
		Loggeur.writeLog("Serveur", "Création des threads de traitement : " + nbThreads + " threads");
		tabThreads = new Vector<ThreadTraitement>(nbThreads);
		for(int i=1; i<=nbThreads; i++){
			ThreadTraitement thread = new ThreadTraitement(i);
			tabThreads.add(thread);
			thread.start();
		}		
		serveurJeton.start();

		Loggeur.writeLog("Serveur", "Serveur " + serveurInfos.nom + " pret");
	}

	/**
	 * Effectue l'arret du serveur proprement
	 * @throws RemoteException 
	 * @throws NotBoundException  
	 */
	public void destroy() throws RemoteException, NotBoundException {
		
		// Averti l'administrateur que la plateforme est arreté
		if(this.adminInfos != null){
			Loggeur.writeLog("Serveur", "Envoie d'une notification à l'administrateur");
			Registry rAdmin;
			rAdmin = LocateRegistry.getRegistry(adminInfos.hote, adminInfos.port);
			AdminRemote admin = (AdminRemote)rAdmin.lookup(adminInfos.nom);
			Loggeur.writeLog("Serveur", "Connexion a l'administrateur OK");
			admin.alerteFinTerminaison();
			Loggeur.writeLog("Serveur", "Administrateur notifié");	
		}

		// Fermeture de la connexion RMI
		Loggeur.writeLog("Serveur", "Fermeture de la connexion RMI");
		UnicastRemoteObject.unexportObject(this, true);
		Loggeur.writeLog("Serveur", "Unbind du serveur");
		Registry rServeur = LocateRegistry.getRegistry("localhost", serveurInfos.port);
		rServeur.unbind(serveurInfos.nom);
		
		Loggeur.writeLog("Serveur", "Le serveur a été correctement arreté");
		
		System.exit(0);
	}
	
	/**
	 * Etablie la connexion avec le serveur suivant passé en paramètre
	 * @param suivant Informations du serveur suivant
	 * @throws RemoteException
	 * @throws MalformedURLException
	 * @throws NotBoundException
	 * @throws NamingException
	 */
	public void setSuivant(ServeurInfos suivant) throws RemoteException, MalformedURLException, NotBoundException, NamingException {
		Loggeur.writeLog("Serveur", "Connexion au serveur suivant " + suivant);
		Registry rServeur = LocateRegistry.getRegistry(suivant.hote, suivant.port);
		this.suivant = (ServeurRemote)rServeur.lookup(suivant.nom);
		Loggeur.writeLog("Serveur", "Connexion au serveur suivant OK");
	}

	/**
	 * Etablie la connexion avec un client passé en paramètre
	 * @param clientInfos Informations du client
	 * @return Objet distant représentant le client
	 * @throws RemoteException
	 * @throws MalformedURLException
	 * @throws NotBoundException
	 * @throws NamingException
	 */
	private ClientRemote connectClient(ClientInfos clientInfos) throws RemoteException, MalformedURLException, NotBoundException, NamingException {
		Loggeur.writeLog("Serveur", "Connexion au client " + clientInfos);
		Registry rServeur = LocateRegistry.getRegistry(clientInfos.hote, clientInfos.port);
		ClientRemote client = (ClientRemote)rServeur.lookup(clientInfos.nom);
		Loggeur.writeLog("Serveur", "Connexion au client OK");
		return client;
	}

	private synchronized boolean isActive(){
		if(enCoursTraitement){
			return true;
		}
		for(ThreadTraitement th : tabThreads){
			if(th.isOccupe()){
				return true;
			}
		}
		return false;
	}

	/**
	 * @see serveur.ServeurRemote#demanderTraitementImage(ClientInfos)
	 */
	public void demanderTraitementImage(ClientInfos clientInfos) throws RemoteException, ServeurException {
		if(terminaison.isModeTerminaison()){
			throw new ServeurRefuseTraitementException("Plus d'acceptation de traitement");			
		}
		Loggeur.writeLog("Serveur", "Reception d'une demande de traitement de " + clientInfos);
		JetonDemandeTraitement jeton = new JetonDemandeTraitement(serveurInfos, clientInfos);
		Loggeur.writeLog("Serveur", "Creation du jeton de demande de traitement " + jeton.getId() + " et ajout dans la file");
		synchronized (serveurJeton) {
			fileJeton.add(jeton);
			serveurJeton.notify();
		}
	}

	/**
	 * @see serveur.ServeurRemote#demarrerTerminaison(AdminInfos)
	 */
	public void demarrerTerminaison(AdminInfos adminInfos) throws RemoteException, ServeurException {
		Loggeur.writeLog("Serveur", "Reception d'une demande de terminaison de " + adminInfos);
		terminaison.setModeTerminaison(true);
		JetonTerminaison jeton = new JetonTerminaison(this.serveurInfos, adminInfos, SafraTerminaison.COLOR_WHITE, 0);
		Loggeur.writeLog("Serveur", "Creation du jeton de terminaison " + jeton.getId() + " et envoi au suivant");
		this.envoyerJeton(jeton);
	}

	/**
	 * @see serveur.ServeurRemote#envoyerImage(ClientInfos, ImageSerializable)
	 */
	public void envoyerImage(ClientInfos clientInfos, ImageSerializable image, EnumTypeFiltre filtre) throws RemoteException, ServeurException {
		Loggeur.writeLog("Serveur", "Reception d'une image de " + clientInfos);
		if(clientInfos.equals(this.clientEnCours)){
			throw new ServeurRefuseTraitementException("Le serveur est reservé par un autre client");
		}

		Loggeur.writeLog("Serveur", "Decoupage de l'image de " + clientInfos + ", taille de " + image.width + "x" + image.height);

		List<Morceau> lstMorceau;
		try {
			lstMorceau = Decoupage.decoupe(image);
			Loggeur.writeLog("Serveur", "Image découpé en " + lstMorceau.size() + " morceaux");
			for(Morceau morceau : lstMorceau){
				JetonTraitement jeton = new JetonTraitement(serveurInfos, clientInfos, morceau, filtre);
				Loggeur.writeLog("Serveur", "Creation du jeton de traitement " + jeton.getId() + " et ajout dans la file");
				synchronized (serveurJeton) {
					fileJeton.add(jeton);
					serveurJeton.notify();
				}
			}
			clientEnCours = null;
			enCoursTraitement = false;

		} catch (IOException e) {
			throw new ServeurException("Erreur inconnu : " + e.getMessage());
		}
	}

	/**
	 * @see serveur.ServeurRemote#recevoirJeton(Jeton)
	 */
	public void recevoirJeton(Jeton jeton) throws RemoteException, ServeurException {
		synchronized(serveurJeton){
			Loggeur.writeLog("Serveur", "Reception d'un jeton du serveur précedent : " + jeton);
			if(!(jeton instanceof JetonTerminaison)){
				terminaison.doRecv();
			}
			Loggeur.writeLog("Serveur", "Ajout du jeton " + jeton.getId() + " dans la file : 1+" + fileJeton.size() + " jetons");
			fileJeton.add(jeton);
			serveurJeton.notify();
		}
	}

	/**
	 * Transmet un jeton au serveur suivant
	 * @param jeton Jeton a transmettre
	 * @throws RemoteException
	 * @throws ServeurException
	 */
	public void envoyerJeton(Jeton jeton) throws ServeurException {
		Loggeur.writeLog("ServeurJeton", "Transmission du jeton " + jeton.getId() + " au serveur suivant");
		if(!(jeton instanceof JetonTerminaison)){
			terminaison.doSend();
		}
		try{
			suivant.recevoirJeton(jeton);
		}catch(RemoteException e){
			System.out.println(suivant);
			throw new ServeurCommunicationException("[ERREUR] Impossible de se connecter au suivant : " + e.getMessage());
		}
	}

	public ActionTraitementJeton traiterJeton(JetonDemandeTraitement jeton) throws ServeurException {
		if(enCoursTraitement == false){
			Loggeur.writeLog("ServeurJeton", "Accepte la demande du client");
			enCoursTraitement = true;
			clientEnCours = jeton.client;
			ClientRemote client;
			try {
				client = this.connectClient(jeton.client);
				client.accepterDemandeTraitementImage(this.serveurInfos);
				return ActionTraitementJeton.JETON_TRAITE;
			} catch (Exception e) {
				throw new ServeurCommunicationException("[ERREUR] Impossible de se connecter au client " + jeton.client + " : " + e.getMessage());
			}
		}else{
			return ActionTraitementJeton.JETON_PASSE_SUIVANT;
		}
	}

	public ActionTraitementJeton traiterJeton(JetonTraitement jeton) throws ServeurException {
		boolean traite = false;
		ThreadTraitement thread = null;
		synchronized(tabThreads){
			for(ThreadTraitement tmp : tabThreads){
				if(!tmp.isOccupe() && !traite){
					Loggeur.writeLog("ServeurJeton", "Accepte le traitement du client");
					traite = true;
					thread = tmp;
					synchronized (thread) {
						thread.reserverThread(jeton);
						thread.notify();						
					}
				}
			}
		}
		if(traite){
			return ActionTraitementJeton.JETON_TRAITE;
		}else{
			return ActionTraitementJeton.JETON_PASSE_SUIVANT;			
		}
	}

	public ActionTraitementJeton traiterJeton(JetonTerminaison jeton) throws ServeurException {
		terminaison.setModeTerminaison(true);
		
		ActionTraitementJeton res;
		
		if(terminaison.getState() == SafraTerminaison.STATE_PASSIVE){
			if(this.serveurInfos.equals(jeton.initiateur)){
				Loggeur.writeLog("Terminaison", "Initiateur");
				if(jeton.color == SafraTerminaison.COLOR_WHITE && 
						terminaison.getColor() == SafraTerminaison.COLOR_WHITE && 
						terminaison.getMc() + jeton.q == 0){
					Loggeur.writeLog("Serveur", "La terminaison de la plate-forme a été detecté");
					adminInfos = jeton.admin;
					
					JetonStop jeton2 = new JetonStop(this.serveurInfos);
					Loggeur.writeLog("Serveur", "Creation du jeton de stop " + jeton.getId() + " et envoi au suivant");
					this.envoyerJeton(jeton2);
					
					res = ActionTraitementJeton.JETON_TRAITE;
				}else{
					jeton.color = SafraTerminaison.COLOR_WHITE;
					jeton.q = 0;
					res = ActionTraitementJeton.JETON_PASSE_SUIVANT;
				}
			}else{
				if(terminaison.getColor() == SafraTerminaison.COLOR_WHITE){
					jeton.q += terminaison.getMc();
					res = ActionTraitementJeton.JETON_PASSE_SUIVANT;
				}else{
					jeton.color = SafraTerminaison.COLOR_BLACK;
					jeton.q += terminaison.getMc();
					res = ActionTraitementJeton.JETON_PASSE_SUIVANT;
				}
			}
			terminaison.setColor(SafraTerminaison.COLOR_WHITE);
		}else{
			res = ActionTraitementJeton.JETON_PASSE_SUIVANT;	
		}
		
		return res;
	}

	public ActionTraitementJeton traiterJeton(JetonStop jeton) throws ServeurException {
		terminaison.setEndTerminaison(true);

		if(jeton.initiateur.equals(this.serveurInfos)){
			return ActionTraitementJeton.JETON_TRAITE;
		}else{
			return ActionTraitementJeton.JETON_PASSE_SUIVANT;			
		}
	}
	
	public void run(){
		Loggeur.writeLog("ServeurJeton", "Lancement du thread de traitement des jetons");
		try {
			while(!terminaison.isEndTerminaison()){
				synchronized(serveurJeton){
					while(fileJeton.isEmpty()){
						if(terminaison.isModeTerminaison() && !this.isActive()){
							Loggeur.writeLog("ServeurJeton", "Le serveur est inactif pour la terminaison");							
							terminaison.doIdle();
						}
						Loggeur.writeLog("ServeurJeton", "Attente d'un nouveau jeton à traiter");
						serveurJeton.wait();
					}
				}
				Loggeur.writeLog("ServeurJeton", "Encore " + fileJeton.size() +" jetons dans la file");
				Jeton jeton = fileJeton.poll();
				Loggeur.writeLog("ServeurJeton", "Traitement du jeton " + jeton.getId());
				try {
					ActionTraitementJeton res = jeton.traiterJeton(this);
					if(res == ActionTraitementJeton.JETON_PASSE_SUIVANT){
						this.envoyerJeton(jeton);
					}else if(res == ActionTraitementJeton.JETON_REMET_FILE ){
						Loggeur.writeLog("ServeurJeton", "Remise du jeton " + jeton.getId() + " dans la file");
						fileJeton.add(jeton);
					}
				} catch (ServeurException e) {
					Loggeur.writeLog("ServeurJeton", "Remise du jeton " + jeton.getId() + " dans la file");
					fileJeton.add(jeton);
				}			
			}
			
			Loggeur.writeLog("ServeurJeton", "Fin du serveur de jeton");
			this.destroy();
			
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		Loggeur.writeLog("Serveur", "Arret du serveur !");
		System.exit(0);
		
	}
	
}
