package server;

import javax.jms.*;

/**
 * Classe de gestion de la queue de demande de certificat
 * @author cmoutenet
 */
public class QueueCertificatServeur implements MessageListener {

	private final MessageProducer producer;
	private final MessageConsumer consumer;
	private final QueueSession queueSession;
	private final CertificatServeur serveur;
	private final Queue queue;
	private final QueueConnectionFactory qcf;
	private final QueueConnection qc;

	/**
	 * Constructeur
	 * @param serveur Référence au serveur de certificat
	 * @param queue Queue du serveur de certificat
	 * @param qcf 
	 * @throws JMSException
	 */
	public QueueCertificatServeur(CertificatServeur serveur, Queue queue,
			QueueConnectionFactory qcf) throws JMSException {
		this.serveur = serveur;
		this.queue = queue;
		this.qcf = qcf;
		qc = qcf.createQueueConnection();
		// Mode transactionnelle, avec acquitement manuel
		queueSession = qc.createQueueSession(false, Session.CLIENT_ACKNOWLEDGE);
		consumer = queueSession.createConsumer(queue);
		QueueSession session = qc.createQueueSession(false,
				Session.CLIENT_ACKNOWLEDGE);
		producer = session.createProducer(queue);

		consumer.setMessageListener(this);
		qc.start();
	}

	/**
	 * Envoie un message retournant le certificat demandé dans le Queue
	 * @param certificat Certificat
	 * @throws JMSException
	 */
	public void retournerCertificat(String certificat) throws JMSException {
		CertificatReponse reponse = new CertificatReponse(certificat);
		ObjectMessage msg = queueSession.createObjectMessage();
		msg.setObject(reponse);
		producer.send(msg);
		System.out.println("Retourner le certificat généré ok");
	}

	/**
	 * Evenement de reception d'un message dans la queue
	 */
	public void onMessage(Message msg) {
		//System.out.println("Message " + msg);
		ObjectMessage msgObj = (ObjectMessage) msg;
		try {
			Object obj = msgObj.getObject();
			if (obj instanceof DemandeCertificatClient) {
				String openSSLKey = ((DemandeCertificatClient) obj)
						.getOpenSSLReq();
				serveur.genererCertificatClient(openSSLKey);
				msg.acknowledge();
			} else if (obj instanceof DemandeCertificatServeur) {
				serveur.genererCertificatServeur();
				msg.acknowledge();
			}
		} catch (JMSException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Fermeture de la connexion avec la queue
	 */
	public void close() {
		try {
			qc.close();
		} catch (JMSException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Retourne la référence à la queue
	 * @return
	 */
	public Queue getQueue() {
		return queue;
	}

	/**
	 * Retourne la référence à la connexion factory de la queue
	 * @return
	 */
	public QueueConnectionFactory getQueueConnexionFactory() {
		return qcf;
	}

}
