package client;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.net.MalformedURLException;
import java.rmi.NotBoundException;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;
import java.rmi.server.UnicastRemoteObject;
import java.util.Vector;

import javax.imageio.ImageIO;

import decoupage.ImageSerializable;
import decoupage.Morceau;
import filtre.EnumTypeFiltre;

import serveur.exceptions.ServeurCommunicationException;
import serveur.exceptions.ServeurException;
import serveur.exceptions.ServeurRefuseTraitementException;
import serveur.ServeurInfos;
import serveur.ServeurRemote;
import tools.Loggeur;

/**
 * 
 * @author Beuque Eric
 *
 */
public class Client extends UnicastRemoteObject implements ClientRemote {

	private ClientInfos clientInfos;
	private ServeurRemote serveur;
	private ServeurRemote serveurTraitement;

	private ImageSerializable image;
	private EnumTypeFiltre filtre;

	private Vector<ClientActionListener> lstCal;
	Integer nbMorceauRecu;

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * Constructeur qui ne comprend que le premier serveur
	 * @param idserveur Le nom du serveur
	 * @throws RemoteException
	 */
	public Client(ClientInfos infos) throws RemoteException {
		super();
		this.clientInfos = infos;
		serveur = null;
		image = null;
		filtre = null;
		lstCal = new Vector<ClientActionListener>();
	}

	public void addClientActionListener(ClientActionListener cal){
		if(!lstCal.contains(cal)){
			lstCal.add(cal);
		}
	}
	
	public void init() throws RemoteException {
		Loggeur.writeLog("Client", "Initialisation");
		Loggeur.writeLog("Client", "Bind du client " + clientInfos);
		Registry rServeur = LocateRegistry.getRegistry("localhost", clientInfos.port);
		rServeur.rebind(clientInfos.nom, this);
		Loggeur.writeLog("Client", "Client " + clientInfos.nom + " pret");
	}

	public void destroy() throws RemoteException, NotBoundException {
		Loggeur.writeLog("Client", "Fermeture de la connexion RMI");
		UnicastRemoteObject.unexportObject(this, true);
		Loggeur.writeLog("Client", "Unbind du client");
		Registry rServeur = LocateRegistry.getRegistry("localhost", clientInfos.port);
		rServeur.unbind(clientInfos.nom);
	}
	
	public void removeClientActionListener(ClientActionListener cal){
		if(lstCal.contains(cal)){
			lstCal.remove(cal);
		}
	}

	public void setServeur(ServeurInfos serveurInfos) throws RemoteException, MalformedURLException, NotBoundException {
		Loggeur.writeLog("Client", "Connexion au serveur " + serveurInfos);
		Registry rServeur = LocateRegistry.getRegistry(serveurInfos.hote, serveurInfos.port);
		this.serveur = (ServeurRemote)rServeur.lookup(serveurInfos.nom);
		Loggeur.writeLog("Client", "Connexion au serveur OK");
	}

	public void setTraitement(EnumTypeFiltre filtre){
		this.filtre = filtre;
	}
	
	public void readImage(String fichier) throws IOException, InterruptedException {
		Loggeur.writeLog("Client", "Chargement de l'image : " + fichier);
		BufferedImage img = ImageIO.read(new java.io.File(fichier));
		this.readImage(img);
	}

	public void readImage(BufferedImage img) throws IOException, InterruptedException {
		this.image = new ImageSerializable(img, img.getWidth(), img.getHeight());
		Loggeur.writeLog("Client", "L'image a une taille de " + image.width + "x" + image.height + " pixels");
	}

	public void accepterDemandeTraitementImage(ServeurInfos serveurTraitementInfos) throws RemoteException, ServeurException {
		Loggeur.writeLog("Client", "Connexion au serveur de traitement d'image " + serveurTraitementInfos);
		Registry rServeur = LocateRegistry.getRegistry(serveurTraitementInfos.hote, serveurTraitementInfos.port);
		try {
			this.serveurTraitement = (ServeurRemote)rServeur.lookup(serveurTraitementInfos.nom);
		} catch (NotBoundException e) {
			throw new ServeurCommunicationException("[ERREUR] Impossible de joindre le serveur de traitement " + serveurTraitementInfos + " : " + e.getMessage());
		}
		Loggeur.writeLog("Client", "Connexion au serveur de traitement OK");

		Loggeur.writeLog("Client", "Envoi de l'image au serveur de traitement");
		nbMorceauRecu = 0;
		serveurTraitement.envoyerImage(this.clientInfos, this.image, this.filtre);
	}

	public void recupererTraitementImage(Morceau morceau) throws RemoteException, ServeurException {
		boolean fin = false;
		synchronized(nbMorceauRecu){
			nbMorceauRecu++;
			Loggeur.writeLog("Client", "Recuperation d'un morceau de l'image : "
					+ nbMorceauRecu + "/" + morceau.getNum() + "/" + morceau.getMax() + " "
					+ "[" + morceau.getStartX() + "," + morceau.getStartY() + "]"
					+ "[" + (morceau.getStartX() + morceau.getWidth()) + "," + (morceau.getStartY() + morceau.getHeight()) + "]");
			if(morceau.getMax() == nbMorceauRecu){
				fin = true;
			}
		}
		for(ClientActionListener cal : lstCal){
			cal.onReceptionMorceau(morceau);
		}
		if(fin){
			for(ClientActionListener cal : lstCal){
				cal.onFinTraitement();
			}
		}
	}

	public void traitementImage() throws RemoteException, MalformedURLException, NotBoundException {
		Loggeur.writeLog("Client", "Demande de traitement d'image au serveur");
		try{
			serveur.demanderTraitementImage(clientInfos);
		}catch(ServeurRefuseTraitementException e){
			System.out.println("[ERREUR] Demande de traitement d'image refusé, abandon");
		}catch(ServeurException e){
			System.out.println("[ERREUR] Erreur inconnu venant du serveur");
		}
	}
}
