/*************************************************************************
 *  Compilation:  javac Fibonacci.java
 *  Execution:    java Fibonacci N
 *
 *  Computes and prints the first N Fibonacci numbers.
 *
 *  WARNING:  this program is spectacularly inefficient and is meant
 *            to illustrate a performance bug, e.g., set N = 45.
 *
 *
 *   % java Fibonacci 7
 *   1: 1
 *   2: 1
 *   3: 2
 *   4: 3
 *   5: 5
 *   6: 8
 *   7: 13
 *
 *   Remarks
 *   -------
 *    - The 93rd Fibonacci number would overflow a long, but this
 *      will take so long to compute with this function that we
 *      don't bother to check for overflow.
 *
 *************************************************************************/

import java.io.IOException;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.util.Date;
import java.util.Properties;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

public class FibonacciJavaNoThreadRMI {
	public static void main(String[] args) throws IOException,
			InterruptedException, NamingException {
		int nFibo;
		int nbThread;

		if (args.length < 2) {
			System.out.println("USAGE : n_fibo nb_threads\n"
					+ "        n_fibo : an integer\n"
					+ "        n_threads : an integer\n");
			return;
		}

		nFibo = Integer.parseInt(args[0]);
		nbThread = Integer.parseInt(args[1]);

		Properties env = new Properties();
		env.load(FibonacciJavaNoThreadRMI.class
				.getResourceAsStream("rmi.properties"));

		String name = env.getProperty("serveurName", "serveur");
		// Create the initial context
		Context ctx = new InitialContext(env);
		// Look up an object
		FibonacciJavaNoThreadRMIServeur objServ = new FibonacciJavaNoThreadRMIServeur();
		ctx.rebind(name, objServ);

		// Début du chrono
		Date debut = new Date();

		IFibonacciJavaNoThreadRMIServeur fibo = (IFibonacciJavaNoThreadRMIServeur) ctx
		.lookup(name);

		for (int i = 0; i < nbThread; i++) {
			fibo.fib(nFibo);
		}

		// Fin du chrono
		Date duree = new Date(new Date().getTime() - debut.getTime());
		double realtime = (duree.getTime() / 1000.0);
		System.out.println(nFibo + ";" + nbThread + ";JavaNoThreadRMI;"
				+ realtime);
		System.exit(0);
	}

}

class FibonacciJavaNoThreadRMIServeur extends UnicastRemoteObject implements
		IFibonacciJavaNoThreadRMIServeur {
	private static final long serialVersionUID = -4642053095753824240L;

	public FibonacciJavaNoThreadRMIServeur() throws RemoteException {
		super();
	}

	public long fib(int n) throws RemoteException {
		if (n <= 1)
			return n;
		else {
			long res = fib(n - 1) + fib(n - 2);
			return res;
		}
	}

}

interface IFibonacciJavaNoThreadRMIServeur extends Remote {
	public long fib(int n) throws RemoteException;
}
