package jajacode.interpreteur;

import jajacode.jjtree.*;
import jajacode.exception.*;

import memoire.*;
import memoire.exception.*;

public class InterpreteJajaCodeVisitor implements JajaCodeVisitor {
	
	public Object visit(SimpleNode node, Object data) throws JajaCodeVisitorException {
		return new String("Erreur...");
	}
	
	public Object visit(ASTJajaCode node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		// On compare l'adresse du noeud a interpreter a l'adresse du noeud JajaCode courant
		// Si egal -1, on a atteint la fin du programme
		if(idata.compareAdresse(-1)==0){
			return false;
		}
		// Si egal, on interprete, l'instruction du noeud courant
		if(idata.compareAdresse(((ASTJCNbre)node.jjtGetChild(0)).getValeur())==0){
			return node.jjtGetChild(1).jjtAccept(this,data);
		}else{
			return node.jjtGetChild(2).jjtAccept(this,data);
		}
	}
	
	/**
	 * Aucune operation sur ce noeud
	 * @param node Noeud de l'instruction
	 * @param data Memoire de l'interpretation 
	 */
	public Object visit(ASTJCNil node, Object data) throws JajaCodeVisitorException {
		return null;
	}
	
	/**
	 * Effectue l'instruction JajaCode "init" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTInit node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			idata.memoire.creer();
			idata.appendAdresse(1);
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "swap" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTSwap node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q1 = idata.memoire.depiler();
			Qua q2 = idata.memoire.depiler();
			idata.memoire.empiler(q1);
			idata.memoire.empiler(q2);
			idata.appendAdresse(1);
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "new" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTNew node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			String obj = ((ASTJCSorte)node.jjtGetChild(2)).getValeur();
			if(obj.equals("var")){
				idata.memoire.identVal(((ASTJCIdent)node.jjtGetChild(0)).getValeur(), ((ASTJCType)node.jjtGetChild(1)).getValeur(), ((ASTJCNbre)node.jjtGetChild(3)).getValeur());
			}
			if(obj.equals("meth")){
				Qua q = idata.memoire.depiler();
				if(q.getId()!=null){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.id n'est pas <null>" );
				}else if(q.getVal()==null){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val est <null>" );
				}else if(!q.getObj().equals("cst")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.obj n'est pas <cst>" );
				}else if(q.getSorte()!=null){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.sorte n'est pas <null>" );
				}else{
					idata.memoire.declMeth(((ASTJCIdent)node.jjtGetChild(0)).getValeur(), q.getVal(), ((ASTJCType)node.jjtGetChild(1)).getValeur());
				}
			}
			if(obj.equals("cst")){
				Qua q = idata.memoire.depiler();
				if(q.getId()!=null){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.id n'est pas <null>" );
				}else if(q.getVal()==null){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val est <null>" );
				}else if(!q.getObj().equals("cst")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.obj n'est pas <cst>" );
				}else if(q.getSorte()!=null){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.sorte n'est pas <null>" );
				}else{
					idata.memoire.declCst(((ASTJCIdent)node.jjtGetChild(0)).getValeur(), q.getVal(), ((ASTJCType)node.jjtGetChild(1)).getValeur());
				}
			}
			idata.appendAdresse(1);
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
	    return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "newarray" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTNewArray node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q = idata.memoire.depiler();
			if(q.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.id n'est pas <null>" );
			}else if(q.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val est <null>" );
			}else if(!q.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.obj n'est pas <cst>" );
			}else if(q.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.sorte n'est pas <null>" );
			}else{
				idata.memoire.declTab(((ASTJCIdent)node.jjtGetChild(0)).getValeur(), q.getVal(), ((ASTJCType)node.jjtGetChild(1)).getValeur());
			}
			idata.appendAdresse(1);
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
	    return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "invoke" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTInvoke node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			idata.memoire.empiler(new Qua(null,idata.getAdresse()+1,"cst",null));
			idata.setAdresse(((Integer)idata.memoire.val(((ASTJCIdent)node.jjtGetChild(0)).getValeur())).intValue());
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "return" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTReturn node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q = idata.memoire.depiler();
			if(q.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.id n'est pas <null>" );
			}else if(q.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val est <null>" );
			}else if(!q.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.obj n'est pas <cst>" );
			}else if(q.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val<" + q.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else{
					idata.setAdresse(((Integer)q.getVal()).intValue());
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "push" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTPush node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			idata.memoire.empiler(new Qua(null,node.jjtGetChild(0).jjtAccept(this,data),"cst",null));
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		idata.appendAdresse(1);
	    return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "pop" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTPop node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			idata.memoire.depiler();
			idata.appendAdresse(1);
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "load" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTLoad node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			idata.memoire.empiler(new Qua(null,idata.memoire.val(((ASTJCIdent)node.jjtGetChild(0)).getValeur()),"cst",null));
			idata.appendAdresse(1);
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "aload" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTALoad node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q = idata.memoire.depiler();
			if(q.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.id n'est pas <null>" );
			}else if(q.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val est <null>" );
			}else if(!q.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.obj n'est pas <cst>" );
			}else if(q.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val<" + q.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else{
					idata.memoire.empiler(new Qua(null,idata.memoire.valT(((ASTJCIdent)node.jjtGetChild(0)).getValeur(), ((Integer)q.getVal()).intValue()),"cst",null));
					idata.appendAdresse(1);
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "store" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTStore node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q = idata.memoire.depiler();
			if(q.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.id n'est pas <null>" );
			}else if(q.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val est <null>" );
			}else if(!q.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.obj n'est pas <cst>" );
			}else if(q.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.sorte n'est pas <null>" );
			}else{
				idata.memoire.affecterVal(((ASTJCIdent)node.jjtGetChild(0)).getValeur(), q.getVal());
				idata.appendAdresse(1);
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "astore" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTAStore node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q1 = idata.memoire.depiler();
			Qua q2 = idata.memoire.depiler();
			
			if(q1.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.id n'est pas <null>" );
			}else if(q1.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val est <null>" );
			}else if(!q1.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.obj n'est pas <cst>" );
			}else if(q1.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.sorte n'est pas <null>" );
			}else if(q2.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.id n'est pas <null>" );
			}else if(q2.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val est <null>" );
			}else if(!q2.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.obj n'est pas <cst>" );
			}else if(q2.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q2.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val<" + q2.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else{
					idata.memoire.affecterValT(((ASTJCIdent)node.jjtGetChild(0)).getValeur(), ((Integer)q2.getVal()).intValue(), q1.getVal());
					idata.appendAdresse(1);
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "if" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTIf node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q = idata.memoire.depiler();
			if(q.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.id n'est pas <null>" );
			}else if(q.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val est <null>" );
			}else if(!q.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.obj n'est pas <cst>" );
			}else if(q.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q.getVal().getClass().getSimpleName().equals("Boolean")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val<" + q.getVal().getClass().getSimpleName() + "> n'est pas du type <Boolean>" );
				}else{
					if(((Boolean)q.getVal()).booleanValue()){
						idata.setAdresse(((ASTJCNbre)node.jjtGetChild(0)).getValeur());
					}else{
						idata.appendAdresse(1);
					}
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "goto" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTGoto node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		idata.setAdresse(((ASTJCNbre)node.jjtGetChild(0)).getValeur());
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "inc" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTInc node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q = idata.memoire.depiler();
			if(q.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.id n'est pas <null>" );
			}else if(q.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val est <null>" );
			}else if(!q.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.obj n'est pas <cst>" );
			}else if(q.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val<" + q.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else{
					String i = ((ASTJCIdent)node.jjtGetChild(0)).getValeur();
					Object vali = idata.memoire.val(i);
					if(!vali.getClass().getSimpleName().equals("Integer")){
						throw new InvalidQuaForInstructionException(instr + " : Impossible car " + i +  " n'est pas du type <Integer>" );
					}else{
						idata.memoire.affecterVal(i, ((Integer)vali).intValue()+((Integer)q.getVal()).intValue());
						idata.appendAdresse(1);
					}
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "ainc" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTAInc node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q1 = idata.memoire.depiler();
			Qua q2 = idata.memoire.depiler();
			
			if(q1.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.id n'est pas <null>" );
			}else if(q1.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val est <null>" );
			}else if(!q1.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.obj n'est pas <cst>" );
			}else if(q1.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.sorte n'est pas <null>" );
			}else if(q2.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.id n'est pas <null>" );
			}else if(q2.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val est <null>" );
			}else if(!q2.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.obj n'est pas <cst>" );
			}else if(q2.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q2.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val<" + q2.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else{
					if(!q1.getVal().getClass().getSimpleName().equals("Integer")){
						throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val<" + q1.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
					}else{
						String i = ((ASTJCIdent)node.jjtGetChild(0)).getValeur();
						int ind = ((Integer)q2.getVal()).intValue();
						Object valti = idata.memoire.valT(i,ind);
						if(!valti.getClass().getSimpleName().equals("Integer")){
							throw new InvalidQuaForInstructionException(instr + " : Impossible car " + i +  " n'est pas du type <Integer>" );
						}else{
							idata.memoire.affecterValT(i, ind, ((Integer)valti).intValue()+((Integer)q1.getVal()).intValue());
							idata.appendAdresse(1);
						}
					}
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "nop" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTNop node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		idata.appendAdresse(1);
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "jcstop" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTJCStop node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		idata.setAdresse(-1);
		return true;
	}
	
	/**
	 * Renvoie la valeur du noeud courant
	 * @param node Noeud de l'instruction
	 * @param data Memoire de l'interpretation 
	 */
	public Object visit(ASTJCIdent node, Object data) throws JajaCodeVisitorException {
		return node.getValeur();
	}
	
	/**
	 * Renvoie la valeur du noeud courant
	 * @param node Noeud de l'instruction
	 * @param data Memoire de l'interpretation 
	 */
	public Object visit(ASTJCType node, Object data) throws JajaCodeVisitorException {
		return node.getValeur();
	}
	
	/**
	 * Renvoie la valeur du noeud courant
	 * @param node Noeud de l'instruction
	 * @param data Memoire de l'interpretation 
	 */
	public Object visit(ASTJCSorte node, Object data) throws JajaCodeVisitorException {
		return node.getValeur();
	}
	
	/**
	 * Renvoie la valeur du noeud courant
	 * @param node Noeud de l'instruction
	 * @param data Memoire de l'interpretation 
	 */
	public Object visit(ASTJCVrai node, Object data) throws JajaCodeVisitorException {
		return true;
	}
	
	/**
	 * Renvoie la valeur du noeud courant
	 * @param node Noeud de l'instruction
	 * @param data Memoire de l'interpretation 
	 */
	public Object visit(ASTJCFaux node, Object data) throws JajaCodeVisitorException {
		return false;
	}
	
	/**
	 * Effectue l'instruction JajaCode "neg" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTNeg node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q = idata.memoire.depiler();
			if(q.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.id n'est pas <null>" );
			}else if(q.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val est <null>" );
			}else if(!q.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.obj n'est pas <cst>" );
			}else if(q.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val<" + q.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else{
					idata.memoire.empiler(new Qua(null,-((Integer)q.getVal()).intValue(),"cst",null));
					idata.appendAdresse(1);
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "not" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTNot node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q = idata.memoire.depiler();
			if(q.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.id n'est pas <null>" );
			}else if(q.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val est <null>" );
			}else if(!q.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.obj n'est pas <cst>" );
			}else if(q.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q.getVal().getClass().getSimpleName().equals("Boolean")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q +  " car Qua.val<" + q.getVal().getClass().getSimpleName() + "> n'est pas du type <Boolean>" );
				}else{
					idata.memoire.empiler(new Qua(null,!((Boolean)q.getVal()).booleanValue(),"cst",null));
					idata.appendAdresse(1);
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "add" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTAdd node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q1 = idata.memoire.depiler();
			Qua q2 = idata.memoire.depiler();
			
			if(q1.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.id n'est pas <null>" );
			}else if(q1.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val est <null>" );
			}else if(!q1.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.obj n'est pas <cst>" );
			}else if(q1.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.sorte n'est pas <null>" );
			}else if(q2.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.id n'est pas <null>" );
			}else if(q2.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val est <null>" );
			}else if(!q2.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.obj n'est pas <cst>" );
			}else if(q2.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q1.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val<" + q1.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else if(!q2.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val<" + q2.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else{
					idata.memoire.empiler(new Qua(null,((Integer)q1.getVal()).intValue()+((Integer)q2.getVal()).intValue(),"cst",null));
					idata.appendAdresse(1);
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "sub" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTSub node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q1 = idata.memoire.depiler();
			Qua q2 = idata.memoire.depiler();
			
			if(q1.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.id n'est pas <null>" );
			}else if(q1.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val est <null>" );
			}else if(!q1.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.obj n'est pas <cst>" );
			}else if(q1.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.sorte n'est pas <null>" );
			}else if(q2.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.id n'est pas <null>" );
			}else if(q2.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val est <null>" );
			}else if(!q2.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.obj n'est pas <cst>" );
			}else if(q2.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q1.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val<" + q1.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else if(!q2.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val<" + q2.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else{
					idata.memoire.empiler(new Qua(null,((Integer)q1.getVal()).intValue()-((Integer)q2.getVal()).intValue(),"cst",null));
					idata.appendAdresse(1);
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "mul" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTMul node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q1 = idata.memoire.depiler();
			Qua q2 = idata.memoire.depiler();
			
			if(q1.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.id n'est pas <null>" );
			}else if(q1.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val est <null>" );
			}else if(!q1.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.obj n'est pas <cst>" );
			}else if(q1.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.sorte n'est pas <null>" );
			}else if(q2.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.id n'est pas <null>" );
			}else if(q2.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val est <null>" );
			}else if(!q2.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.obj n'est pas <cst>" );
			}else if(q2.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q1.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val<" + q1.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else if(!q2.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val<" + q2.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else{
					idata.memoire.empiler(new Qua(null,((Integer)q1.getVal()).intValue()*((Integer)q2.getVal()).intValue(),"cst",null));
					idata.appendAdresse(1);
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "div" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTDiv node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q1 = idata.memoire.depiler();
			Qua q2 = idata.memoire.depiler();
			
			if(q1.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.id n'est pas <null>" );
			}else if(q1.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val est <null>" );
			}else if(!q1.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.obj n'est pas <cst>" );
			}else if(q1.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.sorte n'est pas <null>" );
			}else if(q2.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.id n'est pas <null>" );
			}else if(q2.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val est <null>" );
			}else if(!q2.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.obj n'est pas <cst>" );
			}else if(q2.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q1.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val<" + q1.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else if(!q2.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val<" + q2.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else{
					idata.memoire.empiler(new Qua(null,((Integer)q1.getVal()).intValue()/((Integer)q2.getVal()).intValue(),"cst",null));
					idata.appendAdresse(1);
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "cmp" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTCmp node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q1 = idata.memoire.depiler();
			Qua q2 = idata.memoire.depiler();
			
			if(q1.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.id n'est pas <null>" );
			}else if(q1.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val est <null>" );
			}else if(!q1.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.obj n'est pas <cst>" );
			}else if(q1.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.sorte n'est pas <null>" );
			}else if(q2.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.id n'est pas <null>" );
			}else if(q2.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val est <null>" );
			}else if(!q2.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.obj n'est pas <cst>" );
			}else if(q2.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q1.getVal().getClass().getSimpleName().equals(q2.getVal().getClass().getSimpleName()) ){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " de type <" + q1.getVal().getClass().getSimpleName() + "> et le Qua " + q2 + " de type <" + q2.getVal().getClass().getSimpleName() + ">" );
				}else{
					idata.memoire.empiler(new Qua(null,q1.getVal().equals(q2.getVal()),"cst",null));
					idata.appendAdresse(1);
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "sup" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTSup node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q1 = idata.memoire.depiler();
			Qua q2 = idata.memoire.depiler();
			
			if(q1.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.id n'est pas <null>" );
			}else if(q1.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val est <null>" );
			}else if(!q1.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.obj n'est pas <cst>" );
			}else if(q1.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.sorte n'est pas <null>" );
			}else if(q2.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.id n'est pas <null>" );
			}else if(q2.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val est <null>" );
			}else if(!q2.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.obj n'est pas <cst>" );
			}else if(q2.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q1.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val<" + q1.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else if(!q2.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val<" + q2.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else{
					idata.memoire.empiler(new Qua(null,((Integer)q1.getVal()).intValue()>((Integer)q2.getVal()).intValue(),"cst",null));
					idata.appendAdresse(1);
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "inf" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTInf node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q1 = idata.memoire.depiler();
			Qua q2 = idata.memoire.depiler();
			
			if(q1.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.id n'est pas <null>" );
			}else if(q1.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val est <null>" );
			}else if(!q1.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.obj n'est pas <cst>" );
			}else if(q1.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.sorte n'est pas <null>" );
			}else if(q2.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.id n'est pas <null>" );
			}else if(q2.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val est <null>" );
			}else if(!q2.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.obj n'est pas <cst>" );
			}else if(q2.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q1.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val<" + q1.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else if(!q2.getVal().getClass().getSimpleName().equals("Integer")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val<" + q2.getVal().getClass().getSimpleName() + "> n'est pas du type <Integer>" );
				}else{
					idata.memoire.empiler(new Qua(null,((Integer)q1.getVal()).intValue()<((Integer)q2.getVal()).intValue(),"cst",null));
					idata.appendAdresse(1);
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "or" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTOr node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q1 = idata.memoire.depiler();
			Qua q2 = idata.memoire.depiler();
			
			if(q1.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.id n'est pas <null>" );
			}else if(q1.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val est <null>" );
			}else if(!q1.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.obj n'est pas <cst>" );
			}else if(q1.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.sorte n'est pas <null>" );
			}else if(q2.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.id n'est pas <null>" );
			}else if(q2.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val est <null>" );
			}else if(!q2.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.obj n'est pas <cst>" );
			}else if(q2.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q1.getVal().getClass().getSimpleName().equals("Boolean")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val<" + q1.getVal().getClass().getSimpleName() + "> n'est pas du type <Boolean>" );
				}else if(!q2.getVal().getClass().getSimpleName().equals("Boolean")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val<" + q2.getVal().getClass().getSimpleName() + "> n'est pas du type <Boolean>" );
				}else{
					idata.memoire.empiler(new Qua(null,((Boolean)q1.getVal()).booleanValue() || ((Boolean)q2.getVal()).booleanValue(),"cst",null));
					idata.appendAdresse(1);
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Effectue l'instruction JajaCode "and" sur la memoire d'interpretation :
	 * Memoire (Pile, Tas...) + Adresse courante
	 * @param node Noeud de l'instruction
	 * @param data Donnee pour l'interpretation (Memoire + Adresse)
	 * @return Renvoi Vrai si l'operation s'est bien deroule
	 * @throws JajaCodeVisitorException
	 */
	public Object visit(ASTAnd node, Object data) throws JajaCodeVisitorException {
		InterpreteurJajaCodeData idata = (InterpreteurJajaCodeData)data;
		String instr = "Instruction <" + (String)node.jjtAccept(new ToStringJajaCodeInstrVisitor(),data) + "> a la ligne " + idata.getAdresse();
		try{
			Qua q1 = idata.memoire.depiler();
			Qua q2 = idata.memoire.depiler();
			
			if(q1.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.id n'est pas <null>" );
			}else if(q1.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val est <null>" );
			}else if(!q1.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.obj n'est pas <cst>" );
			}else if(q1.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.sorte n'est pas <null>" );
			}else if(q2.getId()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.id n'est pas <null>" );
			}else if(q2.getVal()==null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val est <null>" );
			}else if(!q2.getObj().equals("cst")){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.obj n'est pas <cst>" );
			}else if(q2.getSorte()!=null){
				throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.sorte n'est pas <null>" );
			}else{
				if(!q1.getVal().getClass().getSimpleName().equals("Boolean")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q1 +  " car Qua.val<" + q1.getVal().getClass().getSimpleName() + "> n'est pas du type <Boolean>" );
				}else if(!q2.getVal().getClass().getSimpleName().equals("Boolean")){
					throw new InvalidQuaForInstructionException(instr + " : Impossible avec le Qua " + q2 +  " car Qua.val<" + q2.getVal().getClass().getSimpleName() + "> n'est pas du type <Boolean>" );
				}else{
					idata.memoire.empiler(new Qua(null,((Boolean)q1.getVal()).booleanValue() && ((Boolean)q2.getVal()).booleanValue(),"cst",null));
					idata.appendAdresse(1);
				}
			}
		}catch(MemoireException e){
			throw new JajaCodeMemoireException(instr + " : " + e.toString());
		}
		return true;
	}
	
	/**
	 * Renvoie la valeur du noeud courant
	 * @param node Noeud de l'instruction
	 * @param data Memoire de l'interpretation 
	 */
	public Object visit(ASTJCNbre node, Object data) throws JajaCodeVisitorException {
		return node.getValeur();
	}
	
}
