%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% Programme PROLOG d'un joueur de SIAM
%%% 
%%% Date : 2008/05/20
%%%
%%% Auteur : Beuque Eric, Etienne Christophe
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%% PREDICATS DE BASE %%%%%%%%%%%%%%%%%%%%%%%%%

% case(TAB, LC, C)/3
% TAB : le plateau de jeu
% LC  : coordonnee de la case
% C   : contenu de la case au coordonnee LC dans TAB
% Renvoi YES, si le contenu de la case de coordonnee LC par rapport
% au plateau TAB correspond a C.
case([[C,_,_,_,_],_,_,_,_], [1,1], C).
case([[_,C,_,_,_],_,_,_,_], [1,2], C).
case([[_,_,C,_,_],_,_,_,_], [1,3], C).
case([[_,_,_,C,_],_,_,_,_], [1,4], C).
case([[_,_,_,_,C],_,_,_,_], [1,5], C).
case([_,[C,_,_,_,_],_,_,_], [2,1], C).
case([_,[_,C,_,_,_],_,_,_], [2,2], C).
case([_,[_,_,C,_,_],_,_,_], [2,3], C).
case([_,[_,_,_,C,_],_,_,_], [2,4], C).
case([_,[_,_,_,_,C],_,_,_], [2,5], C).
case([_,_,[C,_,_,_,_],_,_], [3,1], C).
case([_,_,[_,C,_,_,_],_,_], [3,2], C).
case([_,_,[_,_,C,_,_],_,_], [3,3], C).
case([_,_,[_,_,_,C,_],_,_], [3,4], C).
case([_,_,[_,_,_,_,C],_,_], [3,5], C).
case([_,_,_,[C,_,_,_,_],_], [4,1], C).
case([_,_,_,[_,C,_,_,_],_], [4,2], C).
case([_,_,_,[_,_,C,_,_],_], [4,3], C).
case([_,_,_,[_,_,_,C,_],_], [4,4], C).
case([_,_,_,[_,_,_,_,C],_], [4,5], C).
case([_,_,_,_,[C,_,_,_,_]], [5,1], C).
case([_,_,_,_,[_,C,_,_,_]], [5,2], C).
case([_,_,_,_,[_,_,C,_,_]], [5,3], C).
case([_,_,_,_,[_,_,_,C,_]], [5,4], C).
case([_,_,_,_,[_,_,_,_,C]], [5,5], C).

% case_valide(LC)/1
% LC : coordonnee de la case
% Renvoi YES, si les coordonnees LC represente une case valide
% dans le plateau.
case_valide([L,C]) :-
	L >= 1,
	L =< 5,
	C >= 1,
	C =< 5.

% case_suivante(LC, ORIEN, ?NLC)/3
% LC    : coordonnee de la case d'origine
% ORIEN : orientation dans laquelle calculer la case suivante
% NLC   : coordonnee de la nouvelle case
% Calcule les coordonnees de la case suivante NLC,
% a partir de la case [L, C], dans le sens ORIEN.
case_suivante([L,C], h, [LN, C]) :-
	LN is L-1.
case_suivante([L,C], d, [L, CN]) :-
	CN is C+1.
case_suivante([L,C], b, [LN, C]) :-
	LN is L+1.
case_suivante([L,C], g, [L, CN]) :-
	CN is C-1.

% bordure(LC)/1
% LC : coordonnee de la case
% Renvoi YES si la case de coordonnee LC est situe en bordure
% du plateau.
bordure([1,_]) :- !.
bordure([5,_]) :- !.
bordure([_,1]) :- !.
bordure([_,5]) :- !.

% orien_oppose(ORIEN, OORIEN)/2
% ORIEN : une orientaion
% OORIEN : l'orientation opposee de ORIEN
% Renvoi YES si l'orientation ORIEN, est l'opposee de l'orientation OORIEN.
orien_oppose(h, b).
orien_oppose(d, g).
orien_oppose(b, h).
orien_oppose(g, d).

% depl_orien(LCD, LCA, ORIEN)/3
% LCD   : coordonnee de la case de depart
% LCA   : coordonnee de la case d'arrivee
% ORIEN : orientation pour aller de LCD a LCA
% Renvoi YES si pour rejoindre la case LCA depuis la case
% LCD necessite un deplacement dans le sens ORIEN.
depl_orien([L,C1],[L,C2],g) :-
	C1 > C2, !.
depl_orien([L,_],[L,_],d) :- !.
depl_orien([L1,C],[L2,C],h) :-
	L1 > L2, !.
depl_orien([_,C],[_,C],b) :- !.
depl_orien([L1,_],[L2,_],b) :-
	L2 > L1, !.
depl_orien([L1,_],[L2,_],h) :-
	L1 > L2, !.
depl_orien([_,C1],[_,C2],d) :-
	C2 > C1, !.
depl_orien([_,C1],[_,C2],g) :-
	C1 > C2, !.

% dernier_pion(TAB, LC, ORIEN, ?NLC)/4
% TAB   : le plateau de jeu
% LC    : coordonnee de la case d'origine
% ORIEN : orientation de recherche
% NLC   : coordonnee de la case contenant le dernier pion
% Renvoi les coordonnees de la derniere case LNC ayant un pion
% en parcourant les cases dans le sens ORIEN, a partir de la
% case LC sur le plateau TAB.
dernier_pion(_, [0,C], _, [1, C]) :- !. % cas on sort du tableau par le haut
dernier_pion(_, [6,C], _, [5, C]) :- !. % cas on sort du tableau par le bas
dernier_pion(_, [L,0], _, [L, 1]) :- !. % cas on sort du tableau par la gauche
dernier_pion(_, [L,6], _, [L, 5]) :- !. % cas on sort du tableau par la droite
dernier_pion(TAB, [L,C], ORIEN, [LN, CN]) :- % cas ou on tombe sur une case vide
	case(TAB, [L,C], 0),
	orien_oppose(ORIEN, OORIEN),
	case_suivante([L,C], OORIEN, [LN, CN]), !.
dernier_pion(TAB, [L,C], ORIEN, [LN, CN]) :- % cas ou on est sur un pion
	case_suivante([L,C], ORIEN, [L1, C1]),
	dernier_pion(TAB, [L1,C1], ORIEN, [LN, CN]).

% premiere_montagne(TAB, CASE, ORIEN, ?MCASE)/4
% TAB   : le plateau de jeu
% CASE  : coordonnee de la case d'origine
% ORIEN : orientation de recherche
% MCASE : coordonnee de la case contenant la premiere montagne
% Renvoi la case de la premiere montagne MCASE dans la liste de pion
% du plateau TAB a partir de la case CASE dans le sens ORIEN.
premiere_montagne(TAB, CASE, _, CASE) :-
	case(TAB, CASE, m), !.
premiere_montagne(TAB, CASE, ORIEN, MCASE) :-
	\+ case(TAB, CASE, 0),
	case_valide(CASE),
	case_suivante(CASE, ORIEN, NCASE),
	premiere_montagne(TAB, NCASE, ORIEN, MCASE).

% derniere_montagne(TAB, CASE, ORIEN, ?MCASE)/4
% TAB   : le plateau de jeu
% CASE  : coordonnee de la case d'origine
% ORIEN : orientation de recherche
% MCASE : coordonnee de la case contenant la derniere montagne
% Renvoi la case de la derniere montagne MCASE dans la liste de pion
% du plateau TAB a partir de la case CASE dans le sens ORIEN.
derniere_montagne(TAB, CASE, ORIEN, MCASE) :-
	dernier_pion(TAB, CASE, ORIEN, NCASE),
	orien_oppose(ORIEN, OORIEN),
	premiere_montagne(TAB, NCASE, OORIEN, MCASE).

%%%%%%%%%%%%%%%%%%%%% PREDICATS POUR LA LISTE DES PION %%%%%%%%%%%%%%%%%%%%%

% liste_pion(TAB, JOUEUR, LPION)/3
% Recupere la liste des pions dans LPION du joueur JOUEUR
% avec leur position sur le plateau TAB
liste_pion(TAB, JOUEUR, LPION) :-
	liste_pion_jeu(TAB, [1,1], JOUEUR, LSPION), % les pions dans le jeu
	liste_pion_horsjeu(LSPION, JOUEUR, LPION). % les pions hors du jeu


% liste_pion_jeu(TAB, CASE, JOUEUR, ?LPION)/4
% Ajoute un pion dans LPION, si le pion contenu dans la case CASE du plateau TAB
% est un pion du joueur JOUEUR. Puis continu de parcourir le tableau jusqu'a la fin.
liste_pion_jeu(_, [6,_], _, []). % des qu'on sort du tableau
liste_pion_jeu([[]|TAB], [L,6], JOUEUR, LPION) :- % on arrive a une fin de colonne
	L < 6,
	L2 is L + 1,
	liste_pion_jeu(TAB, [L2,1], JOUEUR, LPION).
liste_pion_jeu([[CASE|LIG]|TAB], [L,C], JOUEUR, [[JOUEUR,[L,C],ORIEN]|LPION]) :- % case avec pion du joueur
	L < 6,
	C < 6,
	CASE = [JOUEUR, ORIEN],
	C2 is C + 1,
	liste_pion_jeu([LIG|TAB], [L,C2], JOUEUR, LPION).
liste_pion_jeu([[CASE|LIG]|TAB], [L,C], JOUEUR, LPION) :- % case sans pion du joueur
	L < 6,
	C < 6,
	\+ CASE = [JOUEUR, _],
	C2 is C + 1,
	liste_pion_jeu([LIG|TAB], [L,C2], JOUEUR, LPION).

% liste_pion_horsjeu(LSPION, JOUEUR, LPION)/4
% Renvoi la liste des pions LSPION dans LPION avec un pion du JOUEUR qui
% n'est pas sur le plateau si le nombre de pion du joueur JOUEUR dans
% LSPION inferieur a 5.
liste_pion_horsjeu(LSPION, JOUEUR, LPION) :-
	length(LSPION, NB),
	NB < 5,
	LPION = [[JOUEUR,[null,null],null]|LSPION], !.
liste_pion_horsjeu(LPION, _, LPION).

%%%%%%%%%%%%%%%%% PREDICATS POUR LA GENERATION DES COUPS %%%%%%%%%%%%%%%%%%

% Renvoi la force F cumule des pions, calcule a partir du plateau TAB,
% en partant de la case [L,C] suivant l'orientation ORIEN
force(TAB, [L,C], ORIEN, F) :-
	force(TAB, [L,C], ORIEN, 0, 1, F).
force(_, [0,_], _, _, F, F) :- !. % cas ou on sort du tableau par le haut
force(_, [6,_], _, _, F, F) :- !. % cas ou on sort du tableau par le bas
force(_, [_,0], _, _, F, F) :- !. % cas ou on sort du tableau par la gauche
force(_, [_,6], _, _, F, F) :- !. % cas ou on sort du tableau par la droite
force(TAB, [L,C], _, _, F, F) :- % cas ou la case n'a pas de pion
	case(TAB, [L,C], 0), !.
force(_, _, _, _, 0, 0) :- !. % cas ou les pions en sens oppose sont plus fort
force(TAB, [L,C], ORIEN, 0, F, FN) :- % cas ou on est sur la premiere une montagne
	case(TAB, [L,C], m),
	case_suivante([L,C], ORIEN, [LN, CN]),
	force(TAB, [LN,CN], ORIEN, 1, F, FN), !.
force(TAB, [L,C], ORIEN, 1, F, FN) :- % cas ou on est pas sur la premiere montagne
	case(TAB, [L,C], m),
	case_suivante([L,C], ORIEN, [LN, CN]),
	F2 is F - 1,
	force(TAB, [LN,CN], ORIEN, 1, F2, FN), !.
force(TAB, [L,C], ORIEN, MONT, F, FN) :- % cas ou on est sur un pion de meme orientation
	case(TAB, [L,C], [_ , ORIEN]),
	case_suivante([L,C], ORIEN, [LN, CN]),
	F2 is F + 1,
	force(TAB, [LN,CN], ORIEN, MONT, F2, FN), !.
force(TAB, [L,C], ORIEN, MONT, F, FN) :- % cas ou on est sur un pion d'orientation oppos
	case(TAB, [L,C], [_ , ORIEN2]),
	orien_oppose(ORIEN, ORIEN2),
	case_suivante([L,C], ORIEN, [LN, CN]),
	F2 is F - 1,
	force(TAB, [LN,CN], ORIEN, MONT, F2, FN), !.
force(TAB, [L,C], ORIEN, MONT, F, FN) :- % cas ou on est sur un pion d'orientation oppos
	case(TAB, [L,C], [_ , ORIEN2]),
	\+ orien_oppose(ORIEN, ORIEN2),
	case_suivante([L,C], ORIEN, [LN, CN]),
	force(TAB, [LN,CN], ORIEN, MONT, F, FN), !.

% Renvoi dans LNCOUP, le coup COUP avec toutes les orientation possible, ajouta a LCOUP
coup_orien(LCOUP, [JOUEUR, TYPE, [L1,C1], [L2,C2]], LNCOUP) :-
	LNCOUP1 = [[JOUEUR, TYPE, [L1,C1], [L2,C2], h]|LCOUP],
	LNCOUP2 = [[JOUEUR, TYPE, [L1,C1], [L2,C2], d]|LNCOUP1],
	LNCOUP3 = [[JOUEUR, TYPE, [L1,C1], [L2,C2], b]|LNCOUP2],
	LNCOUP = [[JOUEUR, TYPE, [L1,C1], [L2,C2], g]|LNCOUP3].

% Ajoute un coup d'entree a LCOUP dans LNCOUP, si l'entree du PION
% dans la case [L,C] est valide.
coup_entree(TAB, [1,C], [JOUEUR,[null,null],_], LCOUP, LNCOUP) :- % ligne 1
	case(TAB, [1,C], 0),
	coup_orien(LCOUP,[JOUEUR,entree,[null,null],[1,C]],LNCOUP), !.
coup_entree(TAB, [5,C], [JOUEUR,[null,null],_], LCOUP, LNCOUP) :- % ligne 5
	case(TAB, [5,C], 0),
	coup_orien(LCOUP,[JOUEUR,entree,[null,null],[5,C]],LNCOUP), !.
coup_entree(TAB, [L,1], [JOUEUR,[null,null],_], LCOUP, LNCOUP) :- % colonne 1
	case(TAB, [L,1], 0),
	coup_orien(LCOUP,[JOUEUR,entree,[null,null],[L,1]],LNCOUP), !.
coup_entree(TAB, [L,5], [JOUEUR,[null,null],_], LCOUP, LNCOUP) :- % colonne 5
	case(TAB, [L,5], 0),
	coup_orien(LCOUP,[JOUEUR,entree,[null,null],[L,5]],LNCOUP), !.
coup_entree(_, _, _, LCOUP, LCOUP). % case autre cas de figure

% Ajoute un coup de deplacement a LCOUP dans LNCOUP, si le deplacement du PION
% dans la case [L,C] est valide.
coup_deplace(_, _, [_,[null,null],_], LCOUP, LCOUP) :- !. % si le pion est pas dans le jeu
coup_deplace(TAB, [L,C], PION, LCOUP, LNCOUP) :- % deplacement vers le haut
	case(TAB, [L,C], 0),
	PION = [JOUEUR, [LP, CP], _],
	L is LP - 1,
	C is CP,
	coup_orien(LCOUP,[JOUEUR,deplace,[LP,CP],[L,C]],LNCOUP), !.
coup_deplace(TAB, [L,C], PION, LCOUP, LNCOUP) :- % deplacement vers la droite
	case(TAB, [L,C], 0),
	PION = [JOUEUR, [LP, CP], _],
	L is LP,
	C is CP + 1,
	coup_orien(LCOUP,[JOUEUR,deplace,[LP,CP],[L,C]],LNCOUP), !.
coup_deplace(TAB, [L,C], PION, LCOUP, LNCOUP) :- % deplacement vers le bas
	case(TAB, [L,C], 0),
	PION = [JOUEUR, [LP, CP], _],
	L is LP + 1,
	C is CP,
	coup_orien(LCOUP,[JOUEUR,deplace,[LP,CP],[L,C]],LNCOUP), !.
coup_deplace(TAB, [L,C], PION, LCOUP, LNCOUP) :- % deplacement vers la gauche
	case(TAB, [L,C], 0),
	PION = [JOUEUR, [LP, CP], _],
	L is LP,
	C is CP - 1,
	coup_orien(LCOUP,[JOUEUR,deplace,[LP,CP],[L,C]],LNCOUP), !.
coup_deplace(_, _, _, LCOUP, LCOUP).

% Ajoute un coup de deplacement a LCOUP dans LNCOUP, si le changement du PION
% dans la case [L,C] est valide, et different de l'orientation actuelle. 
coup_change(_, [_,[null,null],_], LCOUP, LCOUP) :- !. % si le pion est pas dans le jeu
coup_change([L,C], [JOUEUR,[L,C],h], LCOUP, LNCOUP) :- % si orientation du PION est h
	LNCOUP1 = [[JOUEUR, change, [L,C], [L,C], d]|LCOUP],
	LNCOUP2 = [[JOUEUR, change, [L,C], [L,C], b]|LNCOUP1],
	LNCOUP = [[JOUEUR, change, [L,C], [L,C], g]|LNCOUP2], !.
coup_change([L,C], [JOUEUR,[L,C],d], LCOUP, LNCOUP) :- % si orientation du PION est d
	LNCOUP1 = [[JOUEUR, change, [L,C], [L,C], b]|LCOUP],
	LNCOUP2 = [[JOUEUR, change, [L,C], [L,C], g]|LNCOUP1],
	LNCOUP = [[JOUEUR, change, [L,C], [L,C], h]|LNCOUP2], !.
coup_change([L,C], [JOUEUR,[L,C],b], LCOUP, LNCOUP) :- % si orientation du PION est b
	LNCOUP1 = [[JOUEUR, change, [L,C], [L,C], g]|LCOUP],
	LNCOUP2 = [[JOUEUR, change, [L,C], [L,C], h]|LNCOUP1],
	LNCOUP = [[JOUEUR, change, [L,C], [L,C], d]|LNCOUP2], !.
coup_change([L,C], [JOUEUR,[L,C],g], LCOUP, LNCOUP) :- % si orientation du PION est g
	LNCOUP1 = [[JOUEUR, change, [L,C], [L,C], h]|LCOUP],
	LNCOUP2 = [[JOUEUR, change, [L,C], [L,C], d]|LNCOUP1],
	LNCOUP = [[JOUEUR, change, [L,C], [L,C], b]|LNCOUP2], !.
coup_change(_, _, LCOUP, LCOUP).

% Ajoute un coup de sortie a LCOUP dans LNCOUP, si la sortie du PION
% dans la case [L,C] est valide.
coup_sortie(_, [_,[null,null],_], LCOUP, LCOUP) :- !. % si le pion est pas dans le jeu
coup_sortie([1,C], [JOUEUR,[1,C],_], LCOUP, LNCOUP) :- % ligne 1
	LNCOUP = [[JOUEUR,sortie,[1,C],[null,null],null] | LCOUP], !.
coup_sortie([5,C], [JOUEUR,[5,C],_], LCOUP, LNCOUP) :- % ligne 5
	LNCOUP = [[JOUEUR,sortie,[5,C],[null,null],null] | LCOUP], !.
coup_sortie([L,1], [JOUEUR,[L,1],_], LCOUP, LNCOUP) :- % colonne 1
	LNCOUP = [[JOUEUR,sortie,[L,1],[null,null],null] | LCOUP], !.
coup_sortie([L,5], [JOUEUR,[L,5],_], LCOUP, LNCOUP) :- % colonne 5
	LNCOUP = [[JOUEUR,sortie,[L,5],[null,null],null] | LCOUP], !.
coup_sortie(_, _, LCOUP, LCOUP). % case autre cas de figure

% Ajoute le coup passe en parametre a LCOUP dans LNCOUP,
% si la force cumule des pions fesant opposition est superieur a 0. 
valide_coup_poussee(TAB, [_, _, _, LCA, ORIEN], LCOUP, LCOUP) :- % cas ou la force est = a 0
	force(TAB, LCA, ORIEN, F),
	F =< 0, !.
valide_coup_poussee(_, COUP, LCOUP, [COUP|LCOUP]). % cas ou la force est > a 0

% Ajoute un coup d'entree poussee a LCOUP dans LNCOUP, si la sortie du PION
% dans la case [L,C] est valide.
coup_entreepoussee(TAB, [1,1], [JOUEUR,[null,null],_], LCOUP, LNCOUP) :- % ligne 1, col 1
	case(TAB, [1,1], [_,_]),
	valide_coup_poussee(TAB, [JOUEUR,entreepoussee,[null,null],[1,1],d], LCOUP, LNCOUP1),
	valide_coup_poussee(TAB, [JOUEUR,entreepoussee,[null,null],[1,1],b], LNCOUP1, LNCOUP), !.
coup_entreepoussee(TAB, [5,1], [JOUEUR,[null,null],_], LCOUP, LNCOUP) :- % ligne 5, col 1
	case(TAB, [5,1], [_,_]),
	valide_coup_poussee(TAB, [JOUEUR,entreepoussee,[null,null],[5,1],d], LCOUP, LNCOUP1),
	valide_coup_poussee(TAB, [JOUEUR,entreepoussee,[null,null],[5,1],h], LNCOUP1, LNCOUP), !.
coup_entreepoussee(TAB, [1,5], [JOUEUR,[null,null],_], LCOUP, LNCOUP) :- % ligne 1, col 5
	case(TAB, [1,5], [_,_]),
	valide_coup_poussee(TAB, [JOUEUR,entreepoussee,[null,null],[1,5],b], LCOUP, LNCOUP1),
	valide_coup_poussee(TAB, [JOUEUR,entreepoussee,[null,null],[1,5],g], LNCOUP1, LNCOUP), !.
coup_entreepoussee(TAB, [5,5], [JOUEUR,[null,null],_], LCOUP, LNCOUP) :- % ligne 5, col 5
	case(TAB, [5,5], [_,_]),
	valide_coup_poussee(TAB, [JOUEUR,entreepoussee,[null,null],[5,5],h], LCOUP, LNCOUP1),
	valide_coup_poussee(TAB, [JOUEUR,entreepoussee,[null,null],[5,5],g], LNCOUP1, LNCOUP), !.
coup_entreepoussee(TAB, [1,C], [JOUEUR,[null,null],_], LCOUP, LNCOUP) :- % ligne 1
	case(TAB, [1,C], [_,_]),
	valide_coup_poussee(TAB, [JOUEUR,entreepoussee,[null,null],[1,C],b], LCOUP, LNCOUP), !.
coup_entreepoussee(TAB, [5,C], [JOUEUR,[null,null],_], LCOUP, LNCOUP) :- % ligne 5
	case(TAB, [5,C], [_,_]),
	valide_coup_poussee(TAB, [JOUEUR,entreepoussee,[null,null],[5,C],h], LCOUP, LNCOUP), !.
coup_entreepoussee(TAB, [L,1], [JOUEUR,[null,null],_], LCOUP, LNCOUP) :- % colonne 1
	case(TAB, [L,1], [_,_]),
	valide_coup_poussee(TAB, [JOUEUR,entreepoussee,[null,null],[L,1],d], LCOUP, LNCOUP), !.
coup_entreepoussee(TAB, [L,5], [JOUEUR,[null,null],_], LCOUP, LNCOUP) :- % colonne 5
	case(TAB, [L,5], [_,_]),
	valide_coup_poussee(TAB, [JOUEUR,entreepoussee,[null,null],[L,5],g], LCOUP, LNCOUP), !.
coup_entreepoussee(_, _, _, LCOUP, LCOUP). % case autre cas de figure

% Ajoute un coup de poussee a LCOUP dans LNCOUP, si la poussee du PION
% dans la case [L,C] est valide.
% coup_poussee([[0,0,0,0,0],[0,0,0,0,0],[m,m,[r,g],[r,g],0],[0,m,0,0,0],[0,0,0,0,0]],[3,3],[r,[3,4],g],[],LCOUP).
coup_poussee(_, _, [_,[null,null],_], LCOUP, LCOUP) :- !. % si le pion est pas dans le jeu
coup_poussee(TAB, [L,C], PION, LCOUP, LNCOUP) :- % poussee vers le haut
	\+ case(TAB, [L,C], 0),
	PION = [JOUEUR, [LP, CP], h],
	L is LP - 1,
	C is CP,
	valide_coup_poussee(TAB, [JOUEUR,poussee,[LP,CP],[L,C],h], LCOUP, LNCOUP), !.
coup_poussee(TAB, [L,C], PION, LCOUP, LNCOUP) :- % poussee vers la droite
	\+ case(TAB, [L,C], 0),
	PION = [JOUEUR, [LP, CP], d],
	L is LP,
	C is CP + 1,
	valide_coup_poussee(TAB, [JOUEUR,poussee,[LP,CP],[L,C],d], LCOUP, LNCOUP), !.
coup_poussee(TAB, [L,C], PION, LCOUP, LNCOUP) :- % poussee vers le bas
	\+ case(TAB, [L,C], 0),
	PION = [JOUEUR, [LP, CP], b],
	L is LP + 1,
	C is CP,
	valide_coup_poussee(TAB, [JOUEUR,poussee,[LP,CP],[L,C],b], LCOUP, LNCOUP), !.
coup_poussee(TAB, [L,C], PION, LCOUP, LNCOUP) :- % poussee vers la gauche
	\+ case(TAB, [L,C], 0),
	PION = [JOUEUR, [LP, CP], g],
	L is LP,
	C is CP - 1,
	valide_coup_poussee(TAB, [JOUEUR,poussee,[LP,CP],[L,C],g], LCOUP, LNCOUP), !.
coup_poussee(_, _, _, LCOUP, LCOUP).

% Liste coup possible qui amene un pion dans une case
liste_coups_possible_case_pion(_, _, [], LCOUP, LCOUP).
liste_coups_possible_case_pion(TAB, [L,C], [PION|LPION], LCOUP, LNCOUP) :-
	coup_entree(TAB, [L,C], PION, LCOUP, LNCOUP1),
	coup_deplace(TAB, [L,C], PION, LNCOUP1, LNCOUP2),
	coup_change([L,C], PION, LNCOUP2, LNCOUP3),
	coup_sortie([L,C], PION, LNCOUP3, LNCOUP4),
	coup_entreepoussee(TAB, [L,C], PION, LNCOUP4, LNCOUP5),
	coup_poussee(TAB, [L,C], PION, LNCOUP5, LNCOUP6),
	liste_coups_possible_case_pion(TAB, [L,C], LPION, LNCOUP6, LNCOUP).

% Renvoi dans LNCOUP, la liste des coups possible sur le plateau TAB
% pour LPION qui aneme un pion la liste LPION a la case [L,C],
% ajoutee a la liste des coups deja calcule LCOUP
liste_coups_possible_case_pions(_, [6,_], _, LCOUP, LCOUP). % des qu'on sort du tableau
liste_coups_possible_case_pions(TAB, [L,6], LPION, LCOUP, LNCOUP) :- % on arrive a une fin de colonne
	L < 6,
	L2 is L + 1,
	liste_coups_possible_case_pions(TAB, [L2,1], LPION, LCOUP, LNCOUP).
liste_coups_possible_case_pions(TAB, [L,C], LPION, LCOUP, LNCOUP) :- % case avec pion du joueur
	L < 6,
	C < 6,
	liste_coups_possible_case_pion(TAB, [L,C], LPION, LCOUP, LNCOUP1),
	C2 is C + 1,
	liste_coups_possible_case_pions(TAB, [L,C2], LPION, LNCOUP1, LNCOUP).

% Renvoie dans LCOUP la liste des coups possible pour
% un joueur JOUEUR sur le plateau TAB
liste_coups_possible_joueur(TAB, JOUEUR, LCOUP) :-
	liste_pion(TAB, JOUEUR, LPION),
	liste_coups_possible_case_pions(TAB, [1,1], LPION, [], LCOUP).

%%%%%%%%%%%%%%%%%%%%%%%%%% HEURISTIQUE %%%%%%%%%%%%%%%%%%%%%%%%%

% Renvoi une NOTE pour le COUP en fonction de l'etat du plateau TAB
% Liste des notes :
% 0 : le coup est un coup perdant
% 1 : le coup est un coup gagnant
% 2 : le coup fait pousser une montagne
% 3 : le coup fait met le pion a cote d'une montagne
% note_coup([[0,0,0,0,0],[0,0,0,0,0],[m,m,[r,g],[r,g],0],[0,m,0,0,0],[0,0,0,0,0]],[r, poussee, [3, 4], [3, 3], g] ,N).
note_coup(TAB, COUP, NOTE) :-
	fin_partie(TAB, COUP, NOTE).

fin_partie(TAB, COUP, NOTE) :-
	COUP = [JOUEUR,entreepoussee,_,LCA,ORIEN],
	derniere_montagne(TAB, LCA, ORIEN, CASE),
	bordure(CASE),
	fin_partie_gagnante(TAB, CASE, ORIEN, JOUEUR, NOTE), !.
fin_partie(TAB, COUP, NOTE) :-
	COUP = [JOUEUR,poussee,_,LCA,ORIEN],
	derniere_montagne(TAB, LCA, ORIEN, CASE),
	bordure(CASE),
	fin_partie_gagnante(TAB, CASE, ORIEN, JOUEUR, NOTE), !.
fin_partie(TAB, COUP, NOTE) :-
	deplace_montagne(TAB, COUP, NOTE).

fin_partie_gagnante(TAB, CASE, ORIEN, JOUEUR, 1) :- % test si le coup est gagnant
	joueur_plus_proche(TAB, CASE, ORIEN, NCASE),
	case(TAB, NCASE, [JOUEUR, _]), !.
fin_partie_gagnante(_, _, _, _, 0). % sinon le coup est perdant

joueur_plus_proche(TAB, CASE, ORIEN, NCASE) :-
	orien_oppose(ORIEN, OORIEN),
	case_suivante(CASE, OORIEN, NCASE),
	case(TAB, NCASE, [_, ORIEN]), !.
joueur_plus_proche(TAB, CASE, ORIEN, NCASE) :-
	orien_oppose(ORIEN, OORIEN),
	case_suivante(CASE, OORIEN, TCASE),
	joueur_plus_proche(TAB, TCASE, ORIEN, NCASE).

deplace_montagne(TAB, COUP, 2) :- % entreepoussee pousse une montagne
	COUP = [_,entreepoussee,_,LCA,ORIEN],
	derniere_montagne(TAB, LCA, ORIEN, MCASE),
	case(TAB, MCASE, m), !.
deplace_montagne(TAB, COUP, 2) :- % poussee pousse une montagne
	COUP = [_,poussee,_,LCA,ORIEN],
	derniere_montagne(TAB, LCA, ORIEN, MCASE),
	case(TAB, MCASE, m), !.
deplace_montagne(TAB, COUP, NOTE) :-
	rapproche_montagne(TAB, COUP, NOTE).

rapproche_montagne(TAB, COUP, 3) :- % deplacement a cote d'une montagne avec orientation en face
	COUP = [_,deplace,_,LCA,ORIEN],	
	case(TAB, CASE, m),
	depl_orien(LCA, CASE, ORIEN),
	distance(LCA, CASE, 1), !.
rapproche_montagne(TAB, COUP, 3) :- % deplacement a cote d'une montagne avec orientation en face
	COUP = [_,deplace, LCD,LCA,ORIEN],
	case(TAB, CASE, m),
	depl_orien(LCA, CASE, ORIEN),
	distance(LCD, CASE, D1),
	distance(LCA, CASE, D2),	
	D1 > D2, !.
rapproche_montagne(TAB, COUP, NOTE) :-
	cas_defaut(TAB, COUP, NOTE).

distance([L1,C1], [L2,C2], D) :-
	D is abs(L1-L2) + abs(C1-C2).

cas_defaut(_, COUP, 94) :-
	COUP = [_,entreepoussee,_,_,_], !.
cas_defaut(_, COUP, 91) :-
	COUP = [_,poussee,_,_,_], !.
cas_defaut(_, COUP, 92) :-
	COUP = [_,deplace,_,_,_], !.
cas_defaut(_, COUP, 90) :-
	COUP = [_,entree,_,_,_], !.
cas_defaut(_, COUP, 93) :-
	COUP = [_,change,_,_,_], !.
cas_defaut(_, COUP, 95) :-
	COUP = [_,sortie,_,_,_], !.

% Renvoi dans MCOUP et MNOTE, le meilleur coup entre COUP et AMCOUP
% en fonction de leur note respective NOTE et AMNOTE
coup_mieux(COUP, 0, _, 0, COUP, 0) :- !.
coup_mieux(_, 0, AMCOUP, AMNOTE, AMCOUP, AMNOTE) :- !.
coup_mieux(COUP, NOTE, _, AMNOTE, COUP, NOTE) :-
	NOTE =< AMNOTE, !.
coup_mieux(_, _, AMCOUP, AMNOTE, AMCOUP, AMNOTE).

% Parcours la liste des coups LCOUP et renvoi dans MCOUP et MNOTE,
% le meilleur coup avec sa note, en fonction de l'actuelle meilleur
% coup AMCOUP et sa note AMNOTE, et aussi de l'etat du plateau TAB
parcours_lcoup(_, [], MCOUP, MNOTE, MCOUP, MNOTE) :- !.
parcours_lcoup(TAB, [COUP|LCOUP], AMCOUP, AMNOTE, NMCOUP, NMNOTE) :-
	note_coup(TAB, COUP, NOTE),
	coup_mieux(COUP, NOTE, AMCOUP, AMNOTE, MCOUP, MNOTE),
	parcours_lcoup(TAB, LCOUP, MCOUP, MNOTE, NMCOUP, NMNOTE).

% Renvoi le meilleur coup dans la liste LCOUP, en fonction
% du plateau TAB et le met dans COUP.
meilleur_coup(_, [], []) :- !.
meilleur_coup(TAB, [COUP|LCOUP], NMCOUP) :-
	note_coup(TAB, COUP, NOTE),
	parcours_lcoup(TAB, LCOUP, COUP, NOTE, MCOUP, MNOTE),
	coup_final(MCOUP, MNOTE, NMCOUP).

coup_final([JOUEUR,poussee,LCD,LCA,_], 0, [JOUEUR,perdu,LCD,LCA,null]) :- !.
coup_final([JOUEUR,_,LCD,LCA,ORIEN], 0, [JOUEUR,perdu,LCD,LCA,ORIEN]) :- !.
coup_final([JOUEUR,poussee,LCD,LCA,_], 1, [JOUEUR,gagne,LCD,LCA,null]) :- !.
coup_final([JOUEUR,_,LCD,LCA,ORIEN], 1, [JOUEUR,gagne,LCD,LCA,ORIEN]) :- !.
coup_final([JOUEUR,poussee,LCD,LCA,_], _, [JOUEUR,poussee,LCD,LCA,null]) :- !.
coup_final(COUP, _, COUP).		

%%%%%%%%%%%%%%%%%%%%%%%%%% PREDICATS FINAL %%%%%%%%%%%%%%%%%%%%%%%%%

% Calcul le meilleur coup suivant a partir d'un etat TAB pour le joueur avec
% les PION (e ou r), et le met dans COUP
% coup_suivant([[0,0,0,0,0],[0,[r,h],0,0,0],[0,m,m,m,0],[0,0,0,[e,b],0],[0,[r,d],0,0,[r,b]]],r,COUP).
coup_suivant(TAB, JOUEUR, COUP) :-
	liste_coups_possible_joueur(TAB, JOUEUR, LCOUP),
	meilleur_coup(TAB, LCOUP, COUP).

l_coup_suivant(TAB, JOUEUR, LCOUP) :-
	liste_coups_possible_joueur(TAB, JOUEUR, LCOUP).

nb_coup_suivant(TAB, JOUEUR, COUP) :-
	liste_coups_possible_joueur(TAB, JOUEUR, LCOUP),
	length(LCOUP, COUP).

n_coup_suivant(TAB, JOUEUR, COUP) :-
	coup_suivant(TAB, JOUEUR, MCOUP),
	note_coup(TAB, MCOUP, MNOTE),
	COUP = [MCOUP, MNOTE].
	
	
